#include <ctime>
#include <cstdlib>
#include <iostream>
#include <deque>
#include <gurobi_c++.h>

using namespace std;

void tsp_mtz(int numNos, bool relaxar);
void tsp_gg(int numNos, bool relaxar);


int main(int argc, char* argv[]) {

	// Escolhe a formulação que será utilizada
	cout << "Formulação:" << endl;
	cout << "[1] Formulação de Miller, Tucker e Zemlin (1960)" << endl;
	cout << "[2] Formulação de Gavish e Graves (1978)" << endl;
	cout << ">> ";

	int formulacao;
	cin >> formulacao;

	// Define o tamanho da formulação
	cout << endl;
	cout << "Número de nós: ";
	cout << ">> ";

	int numNos;
	cin >> numNos;

	// Define se será resolvido a relaxação linear ou o problema inteiro
	cout << endl;
	cout << "Resolver a relaxação linear: " << endl;
	cout << "[1] Não" << endl;
	cout << "[2] Sim" << endl;
	cout << ">> ";

	int relaxar;
	cin >> relaxar;

	// Resolve o problema selecionado com as configurações informadas
	if (formulacao == 1) {
		tsp_mtz(numNos, relaxar == 2);
	} else {
		tsp_gg(numNos, relaxar == 2);
	}

}


// ====================================================================
// Formulação de Miller, Tucker e Zemlin (1960)
// ====================================================================

void tsp_mtz(int numNos, bool relaxar) {

	// ====================================================================
	// Inicializa o gerador de números aleatórios
	// ====================================================================
	srand(1);


	// ====================================================================
	// Cria uma instância aleatória para o TSP
	// ====================================================================
	int n = numNos;										// número de nós
	deque< deque<double> > c(n, deque<double>(n, 0));	// matriz de custos

	// Gera custos aleatórios
	for (int i = 0; i < n; ++i) {
		for (int j = i+1; j < n; ++j) {
			c[i][j] = 1 + (rand() % 100);
			c[j][i] = c[i][j];
		}
	}

	try {

		// ====================================================================
		// Inicializa o modelo do Gurobi
		// ====================================================================

		GRBEnv env = GRBEnv();
		GRBModel model = GRBModel(env);


		// ====================================================================
		// Configura alguns parâmetros do Gurobi
		// ====================================================================

		model.getEnv().set(GRB_IntParam_OutputFlag, 1);			// Exibir log
		model.getEnv().set(GRB_IntParam_Threads, 1);			// Número de threads (0 utiliza todas disponíveis)
		//model.getEnv().set(GRB_IntParam_PrePasses, 0);		// Desabilita o pré-processamento
		//model.getEnv().set(GRB_DoubleParam_Heuristics, 0.0);	// Desabilita o uso de heurísticas


		// ====================================================================
		// Cria as variáveis do problema
		// ====================================================================

		deque< deque<GRBVar> > x;
		for (int i = 0; i < n; ++i) {
			x.push_back(deque<GRBVar>());
			for (int j = 0; j < n; ++j) {
				if (relaxar) {
					GRBVar xij = model.addVar(0.0, 1.0, 0.0, GRB_CONTINUOUS);
					x[i].push_back(xij);
				} else {
					GRBVar xij = model.addVar(0.0, 1.0, 0.0, GRB_BINARY);
					x[i].push_back(xij);
				}
			}
		}

		deque<GRBVar> u;
		for (int i = 0; i < n; ++i) {
			GRBVar ui = model.addVar(1, n-1, 0.0, GRB_CONTINUOUS);
			u.push_back(ui);
		}

		model.update();


		// ====================================================================
		// Define a função objetivo
		// ====================================================================

		GRBLinExpr obj;
		for (int i = 0; i < n; ++i) {
			for (int j = 0; j < n; ++j) {
				obj += c[i][j] * x[i][j];
			}
		}

		model.setObjective(obj, GRB_MINIMIZE);


		// ====================================================================
		// Define as restrições
		// ====================================================================

		for (int i = 0; i < n; ++i) {
			GRBLinExpr constr1;
			for (int j = 0; j < n; ++j) {
				constr1 += x[i][j];
			}
			model.addConstr(constr1 == 1);
		}

		for (int j = 0; j < n; ++j) {
			GRBLinExpr constr2;
			for (int i = 0; i < n; ++i) {
				constr2 += x[i][j];
			}
			model.addConstr(constr2 == 1);
		}

		for (int i = 1; i < n; ++i) {
			for (int j = 1; j < n; ++j) {
				GRBLinExpr constr3;
				constr3 += u[i] - u[j] + (n - 1) * x[i][j];
				model.addConstr(constr3 <= n - 2);
			}
		}


		// ====================================================================
		// Resolve o problema
		// ====================================================================

		model.optimize();


		// ====================================================================
		// Obtém o tour ótimo
		// ====================================================================

		deque<int> solucao;
		solucao.push_back(0);
		int noAtual = 0;

		while (solucao.size() < n) {
			for (int i = 0; i < n; ++i) {
				if (x[noAtual][i].get(GRB_DoubleAttr_X) > 0.5) {
					solucao.push_back(i);
					noAtual = i;
					break;
				}
			}
		}


		// ====================================================================
		// Exibe o resultado obtido
		// ====================================================================

		cout << endl;

		if (!relaxar) {
			cout << "Solução: [ ";
			for (int i = 0; i < solucao.size(); ++i) {
				cout << solucao[i] << " ";
			}
			cout << "]" << endl;
		}

		cout << "Custo: " << model.get(GRB_DoubleAttr_ObjVal) << endl;
		cout << "Tempo: " << model.get(GRB_DoubleAttr_Runtime);


	} catch (GRBException& e) {
		cout << "Código do erro: " << e.getErrorCode() << endl;
		cout << e.getMessage() << endl;
	} catch (...) {
		cout << "Ocorreu uma exceção durante a execução do Gurobi." << endl;
	}

}


// ====================================================================
// Formulação de Gavish e Graves (1978)
// ====================================================================

void tsp_gg(int numNos, bool relaxar) {

	// ====================================================================
	// Inicializa o gerador de números aleatórios
	// ====================================================================
	srand(1);


	// ====================================================================
	// Cria uma instância aleatória para o TSP
	// ====================================================================
	int n = numNos;										// número de nós
	deque< deque<double> > c(n, deque<double>(n, 0));	// matriz de custos

	// Gera custos aleatórios
	for (int i = 0; i < n; ++i) {
		for (int j = i+1; j < n; ++j) {
			c[i][j] = 1 + (rand() % 100);
			c[j][i] = c[i][j];
		}
	}

	try {

		// ====================================================================
		// Inicializa o modelo do Gurobi
		// ====================================================================

		GRBEnv env = GRBEnv();
		GRBModel model = GRBModel(env);


		// ====================================================================
		// Configura alguns parâmetros do Gurobi
		// ====================================================================

		model.getEnv().set(GRB_IntParam_OutputFlag, 1);			// Exibir log
		model.getEnv().set(GRB_IntParam_Threads, 1);			// Número de threads (0 utiliza todas disponíveis)
		//model.getEnv().set(GRB_IntParam_PrePasses, 0);		// Desabilita o pré-processamento
		//model.getEnv().set(GRB_DoubleParam_Heuristics, 0.0);	// Desabilita o uso de heurísticas


		// ====================================================================
		// Cria as variáveis do problema
		// ====================================================================

		deque< deque<GRBVar> > x;
		for (int i = 0; i < n; ++i) {
			x.push_back(deque<GRBVar>());
			for (int j = 0; j < n; ++j) {
				if (relaxar) {
					GRBVar xij = model.addVar(0.0, 1.0, 0.0, GRB_CONTINUOUS);
					x[i].push_back(xij);
				} else {
					GRBVar xij = model.addVar(0.0, 1.0, 0.0, GRB_BINARY);
					x[i].push_back(xij);
				}
			}
		}

		deque< deque<GRBVar> > f;
		for (int i = 0; i < n; ++i) {
			f.push_back(deque<GRBVar>());
			for (int j = 0; j < n; ++j) {
				GRBVar fij = model.addVar(0.0, GRB_INFINITY, 0.0, GRB_CONTINUOUS);
				f[i].push_back(fij);
			}
		}

		model.update();


		// ====================================================================
		// Define a função objetivo
		// ====================================================================

		GRBLinExpr obj;
		for (int i = 0; i < n; ++i) {
			for (int j = 0; j < n; ++j) {
				obj += c[i][j] * x[i][j];
			}
		}

		model.setObjective(obj, GRB_MINIMIZE);


		// ====================================================================
		// Define as restrições
		// ====================================================================

		for (int i = 0; i < n; ++i) {
			GRBLinExpr constr1;
			for (int j = 0; j < n; ++j) {
				constr1 += x[i][j];
			}
			model.addConstr(constr1 == 1);
		}

		for (int j = 0; j < n; ++j) {
			GRBLinExpr constr2;
			for (int i = 0; i < n; ++i) {
				constr2 += x[i][j];
			}
			model.addConstr(constr2 == 1);
		}

		for (int i = 0; i < n; ++i) {
			for (int j = 0; j < n; ++j) {
				GRBLinExpr constr3;
				constr3 += f[i][j] - (n - 1) * x[i][j];
				model.addConstr(constr3 <= 0);
			}
		}

		for (int i = 1; i < n; ++i) {
			GRBLinExpr constr4;
			for (int j = 0; j < n; ++j) {
				if (j != 0) {
					constr4 += f[0][j];
				}
			}
			model.addConstr(constr4 == n - 1);
		}

		for (int j = 1; j < n; ++j) {
			GRBLinExpr constr5;
			for (int i = 0; i < n; ++i) {
				if (i != j) {
					constr5 += f[i][j];
				}
			}
			for (int k = 0; k < n; ++k) {
				if (k != j) {
					constr5 -= f[j][k];
				}
			}
			model.addConstr(constr5 == 1);
		}


		// ====================================================================
		// Resolve o problema
		// ====================================================================

		model.optimize();


		// ====================================================================
		// Obtém o tour ótimo
		// ====================================================================

		deque<int> solucao;
		solucao.push_back(0);
		int noAtual = 0;

		while (solucao.size() < n) {
			for (int i = 0; i < n; ++i) {
				if (x[noAtual][i].get(GRB_DoubleAttr_X) > 0.5) {
					solucao.push_back(i);
					noAtual = i;
					break;
				}
			}
		}


		// ====================================================================
		// Exibe o resultado obtido
		// ====================================================================

		cout << endl;

		if (!relaxar) {
			cout << "Solução: [ ";
			for (int i = 0; i < solucao.size(); ++i) {
				cout << solucao[i] << " ";
			}
			cout << "]" << endl;
		}

		cout << "Custo: " << model.get(GRB_DoubleAttr_ObjVal) << endl;
		cout << "Tempo: " << model.get(GRB_DoubleAttr_Runtime);


	} catch (GRBException& e) {
		cout << "Código do erro: " << e.getErrorCode() << endl;
		cout << e.getMessage() << endl;
	} catch (...) {
		cout << "Ocorreu uma exceção durante a execução do Gurobi." << endl;
	}

}
